function screen_size = screensize(units);
% SCREENSIZE - Find current screen-size
%
% Usage:
% ======
% S = SCREENSIZE, returns the screen-size in pixels, where
%           S = [LEFT,BOTTOM,WIDTH,HEIGHT].
%
% S = SCREENSIZE(UNITS) returns the screen-size in user-specified units of 
%           measurement. Valid values for UNITS are: 'inches', 'centimeters', 
%           'normalized', 'points', 'pixels', and 'characters' (the quotes
%           must be included!).
%
% The coordinates LEFT and BOTTOM correspond with the bottom left point of the
% screen. [LEFT BOTTOM] = [1 1] when expressed in pixel units; for the other
% units [LEFT BOTTOM] = [0 0]. WIDTH and HEIGHT represent the width and height 
% of the screen, expressed in the requested units.

% Variables
% =========
% screen_size  current screen-size, expressed in the requested units
% units        string variable, containing the requested units of measurement
% old_units    original units of measurement


% If the units of measurement are not specified, use pixels
% ---------------------------------------------------------
if nargin == 0
  units = 'pixels';
end

% Retrieve current screen units (will be restored later)
% ------------------------------------------------------
old_units = get(0,'Units');

% Error-catching loop prevents incorrect entries for units
% --------------------------------------------------------
try
    % Set new screen units and obtain screen-size in those units
    % ----------------------------------------------------------
    set(0,'Units',units);
    screen_size = get(0,'ScreenSize');
catch
    disp('-----------------------------------------------------------------');
    disp('Error detected! Most likely, the input-argument for the function,');
    disp('being the ''Units'' specification, was incorrect.');
    disp(' ');
    disp('Valid entries are: ''inches'', ''centimeters'', ''normalized'',');
    disp('''points'', ''pixels'', and ''characters''. The current value was:');
    if ~ischar(units)
    disp(' ');
        disp(['    units = ' num2str(units)]);
        disp('    Note: the input-argument must be a string-variable!');
    else
    disp(' ');
        disp(['    units = ' units]);
    end
    disp(' ');
    disp(['(Here''s the original Matlab error-message: ' lasterr ')']);
    disp('-----------------------------------------------------------------');
    error
end

% Restore old screen units
% ------------------------
set(0,'Units',old_units);

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: December 31, 2004. 
%-----------------------------------------------------------------------